# SL96 functions related to genome composition simulation.

# "setSimGenome" constructs simulated genomes using genotype file.
# Arguments
#	GenoFile: Character string of genotype file name (e.g. "SL96axiom.txt").
#	Bin.size: Bin size of simulated genome (cM).
#	List: If TRUE, simulated genomes are returned as factors in a single list.
# Value
#	List vector for simulated genomes.
#	SimGenomeTable: [[1]]　Matrix of information on simulated genomes. [[2]] Genotype data.
	setSimGenome <- function(GenoFile, Bin.size=0.1, List=FALSE) {
		In <- read.table(GenoFile)
		r <- checkGenoData(In)
		if (r==1) { stop("Format error.") }
		gid <- unique(udac(In[1,5:ncol(In)]))
		g <- read.table(GenoFile, header=TRUE)
		chr <- unique(g$chr)
		Info <- matrix(NA, nrow=length(chr), ncol=2)
		colnames(Info) <- c("chr", "cM")
		SimGenomeTable <- vector(mode="list", length=2)
		SimGenomeTable[[2]] <- g
		z <- data.frame(matrix(NA, nrow=ncol(g)-4, ncol=2))
		colnames(z) <- c("gid", "HapCoode")
		for (i in 1:length(chr)) {
			Info[i,1] <- chr[i]
			cM <- g[g$chr==chr[i],]
			Info[i,2] <- udan(sprintf("%3.1f", max(cM$pos_cM)))+0.1
		}
		g <- g[,!is.element(colnames(g), c("snp_id","chr","pos_bp", "pos_cM"))]
		if (List) {
			TP <- vector(mode="list", length=length(gid))
			count <- 1
			for(i in 1:length(gid)) {
				hap <- vector(mode="list", length=nrow(Info))
				y <- c()
				for (j in 1:2) {
					for(chr in 1:nrow(Info)) { hap[[chr]] <- rep(count, Info[chr,2]/Bin.size) }
					y <- c(y,hap)	
					z[count,1] <- colnames(g)[count]
					z[count,2] <- count
					count <- count + 1
				}
				TP[[i]] <- y
			}
			names(TP) <- gid
			assign("TP", TP, env=.GlobalEnv)
		}
		if (!List) {
			count <- 1
			for(i in 1:length(gid)) {
				hap <- vector(mode="list", length=nrow(Info))
				y <- c()
				for (j in 1:2) {
					for(chr in 1:nrow(Info)) { hap[[chr]] <- rep(count, Info[chr,2]/Bin.size) }
					y <- c(y,hap)	
					z[count,1] <- colnames(g)[count]
					z[count,2] <- count
					count <- count + 1
				}
				assign(gid[i], y, env=.GlobalEnv)
			}
		}
		SimGenomeTable[[1]] <- z
		assign("SimGenomeTable", SimGenomeTable, env=.GlobalEnv)
	}

# "makeProgenies" generates genomes of simulated progenies.
# Arguments
#	Parents1, Parents2: List vector generated using function makeSimGenome (e.g. SL1, SL1.1...).
#	n: Number of progenies to be generated (numeric).
#	Bin.size: Bin size of simulated genome (cM).
# Value
#	List vector of simulated genomes.
	makeProgenies <- function(Parent1, Parent2, n, Bin.size=0.1) {
		if(length(Parent1)==1) Parent1 <- Parent1[[1]]
		if(length(Parent2)==1) Parent2 <- Parent2[[1]]
		CHR.NO <- length(Parent1)/2
		REC.LIST <- vector(mode="list", length=CHR.NO)
		for (C in 1:CHR.NO) {
			REC.POS <- vector(mode="list", length=(n*2))
			CL <- length(unlist(Parent1[[C]]))
			# Recombination frequency setting
			LAMBDA <- CL/(100/Bin.size)
			for (i in 1:length(REC.POS)) {
				x <- round(rpois(1, LAMBDA),0)
				x <- c(0, sort(sample(1:(CL-1), x)), CL)
				REC.POS[[i]] <- x
				}
			REC.LIST[[C]] <- REC.POS
			}
		TENT.POP <- vector(mode="list", length=CHR.NO)	
		for(C in 1:CHR.NO) {
			REC.POS <- REC.LIST[[C]]
			CHROMATIDS <- vector(mode="list", length=n)
			for(i in 1:n) {
				Parent <- Parent1
				PAIRING <- rbind(unlist(Parent[C]), unlist(Parent[C+CHR.NO]))
				R <- sample(1:2, 1)
				if (R%%2==0) PAIRING <- rbind(PAIRING[2,],PAIRING[1,])
				CHROMATID <- c()
				REC.PO <- unlist(REC.POS[[i]])
				for(x in 1:(length(REC.PO)-1)){
					if (x%%2==0) Col <- 1 else Col <- 2
					BLOCK <- PAIRING[Col, (REC.PO[x]+1):(REC.PO[(x+1)])]
					CHROMATID <- c(CHROMATID, BLOCK)
					}
				CHROMATID1 <- as.numeric(CHROMATID)
				Parent <- Parent2
				PAIRING <- rbind(unlist(Parent[C]), unlist(Parent[C+CHR.NO]))
				R <- sample(1:2, 1)
				if (R%%2==0) PAIRING <- rbind(PAIRING[2,],PAIRING[1,])
				CHROMATID <- c()
				REC.PO <- unlist(REC.POS[[i+n]])
				for(x in 1:(length(REC.PO)-1)){
					if (x%%2==0) Col <- 1 else Col <- 2
					BLOCK <- PAIRING[Col, (REC.PO[x]+1):(REC.PO[(x+1)])]
					CHROMATID <- c(CHROMATID, BLOCK)
					}
				CHROMATID2 <- as.numeric(CHROMATID)
				CHROMATIDS[[i]] <- list(CHROMATID1, CHROMATID2)
				}
			TENT.POP[[C]] <- CHROMATIDS
			}
		Population <- vector(mode="list", length=n)
		for (I in 1:n) {
			INDIVIDUAL <- vector(mode="list", length=(CHR.NO*2))
			for (C in 1:CHR.NO) {
				x <- TENT.POP[[C]][I]
				INDIVIDUAL[[C]] <- as.numeric(unlist(x[[1]][1]))
				INDIVIDUAL[[C+CHR.NO]] <- as.numeric(unlist(x[[1]][2]))
				}
			Population[[I]] <- INDIVIDUAL
			}
		Population
		}

# "makeRILs" generates genomes of simulated recombinant inbred lines by using single seed descent.
# Arguments
#	Population: Start population generated by function makeProgenies.
#	Generation: Number of cycles of single seed descend.
#	Bin.size: Bin size of simulated genome (cM).
# Value
#	List vector of simulated genomes.
	makeRILs <- function (Population, Generation, Bin.size=0.1) {
		CHR.NO <- length(Population[[1]])/2
		POP.NO <- length(Population)
		REC.LIST <- vector(mode="list", length=CHR.NO)
		for (C in 1:CHR.NO) {
			REC.POS <- vector(mode="list", length=(POP.NO*2*Generation))
			CL <- length(unlist(Population[[1]][C]))
			LAMBDA <- CL/(100/Bin.size)
			MAX <- round(LAMBDA/0.25)
			for (i in 1:length(REC.POS)) {
				repeat {
					x <- round(rpois(1, LAMBDA),0)
					if (x > MAX) next
					if (length(x) == 1) break
					}
				x <- c(0, sort(sample(1:(CL-1), x)), CL)
				REC.POS[[i]] <- x
				}
			REC.LIST[[C]] <- REC.POS
			}
		for (G in 1:Generation) {
			TENT.POP <- vector(mode="list", length=CHR.NO)
			for(C in 1:CHR.NO) {
				REC.POS <- REC.LIST[[C]]
				CHROMATIDS <- vector(mode="list", length=POP.NO)
				for(i in 1:POP.NO) {
					Parent <- Population[[i]]
					PAIRING <- rbind(unlist(Parent[C]), unlist(Parent[C+CHR.NO]))
					R <- sample(1:2, 1)
					if (R%%2==0) PAIRING <- rbind(PAIRING[2,],PAIRING[1,])
					CHROMATID <- c()
					REC.PO <- unlist(REC.POS[[((G-1)*POP.NO)+i]])
					for(x in 1:(length(REC.PO)-1)){
						if (x%%2==0) Col <- 1 else Col <- 2
						BLOCK <- PAIRING[Col, (REC.PO[x]+1):(REC.PO[(x+1)])]
						CHROMATID <- c(CHROMATID, BLOCK)
						}
					CHROMATID1 <- as.numeric(CHROMATID)	
					R <- sample(1:2, 1)
					if (R%%2==0) PAIRING <- rbind(PAIRING[2,],PAIRING[1,])
					CHROMATID <- c()
					REC.PO <- unlist(REC.POS[[((G-1)*POP.NO)+i+POP.NO]])
					for(x in 1:(length(REC.PO)-1)){
						if (x%%2==0) Col <- 1 else Col <- 2
						BLOCK <- PAIRING[Col, (REC.PO[x]+1):(REC.PO[(x+1)])]
						CHROMATID <- c(CHROMATID, BLOCK)
						}
					CHROMATID2 <- as.numeric(CHROMATID)
					CHROMATIDS[[i]] <- list(CHROMATID1, CHROMATID2)
					}
				TENT.POP[[C]] <- CHROMATIDS
				}
			Population <- vector(mode="list", length=POP.NO)
			for (I in 1:POP.NO) {
				INDIVIDUAL <- vector(mode="list", length=(CHR.NO*2))
				for (C in 1:CHR.NO) {
					x <- TENT.POP[[C]][I]
					INDIVIDUAL[[C]] <- as.numeric(unlist(x[[1]][1]))
					INDIVIDUAL[[C+CHR.NO]] <- as.numeric(unlist(x[[1]][2]))
					}
				Population[[I]] <- INDIVIDUAL
				}
			}
		Population
		}
